### A Pluto.jl notebook ###
# v0.20.4

using Markdown
using InteractiveUtils

# ╔═╡ 235df63c-0786-42f1-a491-017e67a02698
import Pkg; Pkg.activate(".")

# ╔═╡ d5a4a4fc-c52b-42c7-ac4e-387ac95f848b
begin
	using CairoMakie
	using CommonMark
	using CSV
	using DataFrames
	using FITSIO
	using Format
	using LaTeXStrings
	using LombScargle
	using PlutoUI
	using Statistics
end

# ╔═╡ 202d87bd-198a-4394-87d1-a4121daea4cf
md"""
**What is this?**


*This jupyter notebook is part of a collection of notebooks on various topics discussed during the Time Domain Astrophysics course delivered by Stefano Covino at the [Università dell'Insubria](https://www.uninsubria.eu/) in Como (Italy). Please direct questions and suggestions to [stefano.covino@inaf.it](mailto:stefano.covino@inaf.it).*
"""

# ╔═╡ 8353da7d-8182-4a2a-b635-3d9088b3cd24
md"""
**This is a `Julia` notebook**
"""

# ╔═╡ bf2a2828-e952-4e3c-b7bc-aa632262dd30
Pkg.instantiate()

# ╔═╡ fe157ea7-b7c5-4d46-bf1e-a7f69aa8edc3
# ╠═╡ show_logs = false
md"""
$(LocalResource("Pics/TimeDomainBanner.jpg"))
"""

# ╔═╡ 2fe17714-740a-4c8c-8360-b953abf6caf9
md"""
# Science case: Variable stars

***

## Some of the earliest variables discovered: Mira Ceti
***

- David Fabricius discovered “omicron ceti” in 1596:
    - It brightened from the 3rd to the 2nd magnitude, then disappeared. Actually, it disappeared from view since it faded beyond the detection limit.
    - It was re-observed in 1609 by Fabricius, and re-discovered in 1631 by Johann Fokkens Holwarda who determined a 11 month period.
    - Johannes Hevelius observed it in 1639 and 1642, and named the star “Mira” – The Wonderful.


- Earlier ancient discovery records unclear. Hipparchus may have discovered Mira in 134 BC.


## Some of the earliest variables discovered: Algol
***

- Geminiano Montanari discovered “beta Persei” in 1667:
    - Brightness varies from 2.1 to 3.4 magnitude in 2.87 days.
    - Period of less than 3 days credited to John Goodricke in 1782-83.
    - In 1881 Edward Pickering theorized (correctly) that it was an eclipsing binary star system.


- European, Arab, and Chinese cultures knew its variability and considered it evil:
    - “Demon’s Head” – Arab
    - “Mischief-maker” – Arab
    - “Satan’s Head” – Hebrew
    - “Lilith” – Adam’s legendary demonic first wife – Babylon/Hebrew
    - “The  Spectre’s Head” - Europe
    - “Piled-up Corpses” - Chinese


## Some of the earliest variables discovered: Supernovae
***

- Chinese astronomers have been the first (maybe...) to record a supernova:
    - Over 20 historical candidates identified over the past 2000 years. 
    - Confirmed dates include 185, 393, 1006 (brightest and also recorded in Egypt, Iraq, Italy, Japan and Switzerland), 1054 (Crab nebula remnant shown above).


- In 1572 Tycho Brahe observed SN 1572 and argued it was very far from Earth.


"""

# ╔═╡ db391222-d208-42b0-b6f4-eb07e906ad48
# ╠═╡ show_logs = false
md"""
## A rich zoo!
***

- Pulsating Variables:
    - Periodic expansion and contraction of their structure (Cepheids, RR Lyrae, RV Tauri, Long Period, Semi-regular, etc.).

- Eclipsing Variables

- Eruptive Variables:
    - Supernovae, novae, dwarf novae, cataclysmic variables, etc. (we don’t discuss them here).
    
$(LocalResource("Pics/zoo.png"))
"""

# ╔═╡ 588828b5-056f-4deb-a2b2-b76035140359
# ╠═╡ show_logs = false
cm"""
### Variable star properties
***

- Light curves are used to track the luminosity of a variable star with time.
    - Measure period and amplitude

- Some stars are very regular, while others are irregular.

$(LocalResource("Pics/regular.png", :height=>200)) $(LocalResource("Pics/irregular.png", :height=>200)) 

### Intrinsic variables
***

- As giants, stars can become unstable. The temperature and luminosity fluctuate as the gas expands and contracts.
    - These are pulsating variables.
    
$(LocalResource("Pics/intrinsic.png"))

$(LocalResource("Pics/hr.png"))

### Pulsating variables: radial and non-radial modes
***

$(LocalResource("Pics/modes.jpg"))

#### Examples of different types of pulsating variables
***

- Cepheids  Period 1-70 day (on the left) and RV Tauri – Period  up to 100 days (on the right)

$(LocalResource("Pics/rr.gif", :height=>200)) $(LocalResource("Pics/rv.png", :height=>200))

"""

# ╔═╡ 9c7b84ca-6b52-43a0-b861-7dd93a0ad93c
# ╠═╡ show_logs = false
md"""
### Variables as standard candles
***

- The best measure of large distances are variable stars.
    - Pressure and radiation precisely oscillate and luminosity is directly related to the period.
    - The apparent magnitude and period give the distance.
    

$(LocalResource("Pics/standardcandles.png"))

- *RR-Lyrae* stars are low mass pulsating variables. About $1 M_\odot$ with period less than a day often found in clusters.

- Large mass stars that pass through the unstable region are known as *Cepheid variables*. 
    - They have From $5 M_\odot$ to $20 M_\odot$ with period of 1 to 100 days.
    - Cepheids are important to measure distances to other galaxies.
- The pulsanting mwchanism in star’s interior is due to He$^+$ loosing e$^-$ producing He2$^+$. As soon the star expands, it becomes brighter and cooler, and He$_2^+$ + e^$-$ to give He$^+$. And the cycle repeats.


$(LocalResource("Pics/rrhst.png")) $(LocalResource("Pics/cepheidshst.png"))
"""

# ╔═╡ cec98a24-7e9d-4fe7-a716-f6f81d695028
# ╠═╡ show_logs = false
cm"""
### Eclipsing variables
***

$(LocalResource("Pics/eclipsingsystems.png", :height=>180)) $(LocalResource("Pics/eclipsing.gif", :height=>180))

"""

# ╔═╡ 7d835a8b-9075-4d5d-b4cc-ae6cd6b57e27
# ╠═╡ show_logs = false
md"""
### Erupting variables
***

- Erupting variables are stars that have occasional violent outbursts caused by thermonuclear processes either in their surface layers or deep within their interiors.

- Major Types:
    - Supernovae
    - Recurrent Novae
    - Cataclysmic Variables

- Key Properties of CV:
    - Binary Star System.
    - Stellar Material Flows from red dwarf star onto accretion disk surrounding white dwarf companion.
    - Flow stops and starts.
    - Orbital period 78 minutes  up to around 10 hours.
    - “Hot spot” where stream hits accretion disk is often hotter and brighter than either star.

$(LocalResource("Pics/cv.jpg"))
"""

# ╔═╡ b78f97e9-4b2e-4a8f-a692-9520a2aa81f4
md"""
### Let's study a variable source from the ground based LINEAR survey: [11375941](http://simbad.u-strasbg.fr/simbad/sim-id?Ident=linear+11375941&NbIdent=1&Radius=2&Radius.unit=arcmin&submit=submit+id)
***

"""

# ╔═╡ 11d4292c-42eb-46c1-86f2-88107c25895b
begin
	lnr = DataFrame(CSV.File("LINEAR_11375941.csv"))
	    
	first(lnr,5)
end

# ╔═╡ 61b4ac2f-d559-4895-a628-88cc4b94c683
printfmtln("Length of the monitoring: {:.2f} years", (maximum(lnr[!,:t]) - minimum(lnr[!,:t])) / 365)

# ╔═╡ a9125f9d-a9cd-4bfb-95ef-fb8645ae9ff0
begin
	fg1 = Figure()
	
	axfg1 = Axis(fg1[1,1],
	    xlabel="time (MJD)",
	    ylabel="magnitude",
	    title="LINEAR object 11375941"
	    )
	
	scatter!(lnr[!,:t],lnr[!,:mag])
	errorbars!(lnr[!,:t],lnr[!,:mag],lnr[!,:magerr])
	
	axfg1.yreversed=true
	
	fg1
end

# ╔═╡ f73d6e54-031d-4a89-a7a3-3a4c989fc6c6
md"""
- The light-curve shows a very irregular monitoring, likely due to seasonal visiblity of the source
"""

# ╔═╡ f659d524-8f09-4c82-b87e-bf0cabf19e62
begin
	lslnr = lombscargle(lnr[!,:t],lnr[!,:mag],lnr[!,:magerr],nyquist_factor=500,minimum_frequency=0.2)
	
	period_days = 1 ./ lslnr.freq
	period_hours = period_days .* 24
	
	best_period = findmaxperiod(lslnr)
	
	printfmtln("Best period: {:.2f} hours", best_period[1] * 24)
	
	phase = (lnr[!,:t] ./ best_period) .% 1;
end

# ╔═╡ ed624515-a93b-45b7-bcab-ae33144281c2
begin
	fg2 = Figure(size=(1000,400))
	
	ax1fg2 = Axis(fg2[1,1],
	    xlabel="Period (days)",
	    ylabel="Lomb-Scargle Power",
	    title="Lomb-Scargle Periodogram"
	    )
	
	lines!(period_days,lslnr.power)
	
	
	nset_ax = Axis(fg2[1, 1],
	    width=Relative(0.5),
	    height=Relative(0.5),
	    halign=0.9,
	    valign=0.9,
	    xlabel="period (hours)",
	    ylabel="power",
	
	    )
	
	lines!(period_hours,lslnr.power)
	xlims!(1,6)
	
	
	ax2fg2 = Axis(fg2[1,2],
	    xlabel="Phase",
	    ylabel="Magnitude",
	    title="Phased data"
	    )
	
	scatter!(phase,lnr[!,:mag])
	errorbars!(phase,lnr[!,:mag],lnr[!,:magerr])
	
	
	fg2
end

# ╔═╡ f1a6ff9a-d114-4e38-8581-5ed48e78af82
md"""
- The identified period at $\sim 2.58$ hours looks fine and the phased light curve seems consistent with the noise of the data.

- Let's now move to data obtained by a survey carried out with a ornital instrument.

### Let's study a variable source from the space [Kepler](https://en.wikipedia.org/wiki/Kepler_space_telescope) mission: [KIC7198959](http://simbad.u-strasbg.fr/simbad/sim-id?Ident=KIC7198959)
***

"""

# ╔═╡ d820d95f-d627-4ead-8359-60a3c97ed569
begin
	hdu = FITS("kplr007198959-2009259160929_llc.fits")
	
	tbl = DataFrame(hdu[2])
	
	first(tbl,5)
end

# ╔═╡ 1029bc93-7403-4081-b802-5a5664a494c5
md"""
- Among the many columns we are here interested in the "TIME", "SAP_FLUX" and "SAP_FLUX_ERR" columns.
"""

# ╔═╡ 16df63c3-10c7-4c96-8da3-37e3de4293bf
begin
	tblex = tbl[:, filter(x -> (x in ["TIME", "SAP_FLUX", "SAP_FLUX_ERR"]), names(tbl))]
	    
	first(tblex,5)
end

# ╔═╡ eb548a4e-2af3-4060-adbc-b70565294879
md"""
- FITS files usually carry headers full of important information.
"""

# ╔═╡ 58555d35-ff75-4371-b307-7d221f3e7669
headhd = read_header(hdu[2])

# ╔═╡ ef9a0717-9637-4e74-8a49-ed320099fa78
md"""
- Let's remove "NaN" entries from our dataset.
"""

# ╔═╡ b217b22b-b649-4f2c-a347-a632c1cf679d
begin
	allowmissing!(tblex)
	foreach(c->replace!(c,NaN=>missing),eachcol(tblex))
	dropmissing!(tblex)
	
	first(tblex,5)
end

# ╔═╡ 52a3043a-8ca6-49ef-b468-9d6f97c0bdb4
md"""
- Once we have a cleaned time-series let's plot it.
"""

# ╔═╡ bad3dda4-e865-454d-8ffb-a6f34a17addf
begin
	fg3 = Figure()
	
	ax1fg3 = Axis(fg3[1,1],
	    ylabel=L"SAP flux ($10^6 e^-/s$)",
	    title="Observed light curve",
	    xlabel="Time (day)"
	    )
	
	scatter!(tblex[!,:TIME],tblex[!,:SAP_FLUX]/1e6)
	errorbars!(tblex[!,:TIME],tblex[!,:SAP_FLUX]/1e6,tblex[!,:SAP_FLUX_ERR]/1e6)
	    
	xlims!(168, 260)
	
	fg3
end

# ╔═╡ 7da3e53b-75b1-4f88-8a4f-337941a68d2c
md"""
- Undoubtedly, this plot looks somehow weird. We anyawy see there are gaps and a global modulation of brightness of the source.

- Let's now study the observing window.
"""

# ╔═╡ 69e812b8-1ea1-4500-9724-722f2ba6bf65
lsperwin = lombscargle(tblex[!,:TIME],ones(Float64,size(tblex[!,:TIME])),center_data=false,fit_mean=false,minimum_frequency=0,maximum_frequency=200);


# ╔═╡ f7dd869f-5ca0-4586-baff-01b07eeb66c1
begin
	fg4 = Figure()
	
	ax2fg4 = Axis(fg4[1,1],
	    ylabel="Lomb-Scargle power",
	    title="Window Power Spectrum",
	    xlabel=L"Frequency (hour$^{-1}$)"
	    )
	
	lines!(lsperwin.freq / 24, lsperwin.power)
	xlims!(0, 6.5)
	ylims!(-0.1, 1.1)
	text!(0.5, 0.6, text="29.4 minutes")
	arrows!([0.4,1.67], [0.62,0.62], [-0.25,0.25], [0.,0.]) 
	
	
	fg4
end

# ╔═╡ 5a3aaddb-e4bc-48b2-bb1e-6c0a371a03a7
md"""
- We see a regular grid spaced by about 29.4 minutes at the beginning of tye monitoring. However, even indepndently of the gaps, it is not an even sampling as shown below:
"""

# ╔═╡ c7fdf5aa-4981-4ae8-9793-42c848241f34
begin
	sort!(tblex, [:TIME,])
	
	mspl = mean(diff(tblex[!,:TIME])[diff(tblex[!,:TIME]) .< 0.03]) * 24 * 60
	printfmtln("Mean sampling time: {:.2f} minutes", mspl)
	
	fg5 = Figure(size=(1000,400))
	
	ax1fg5 = Axis(fg5[1,1],
	    ylabel=L"$\Delta t$ (min)",
	    title="Time between observations",
	    xlabel="Observation time (days)",
	    yscale=log10
	    )
	
	scatter!(tblex[1:end-1,:TIME],diff(tblex[!,:TIME])*24*60)
	ylims!(10,10000)
	hlines!(mspl)
	
	
	ax2fg5 = Axis(fg5[1,2],
	    ylabel=L"$\Delta t$ - mean sampling time (s)",
	    title="Evolution of the time between observations",
	    xlabel="Observation time (days)",
	    )
	
	scatter!(tblex[1:end-1,:TIME],diff(tblex[!,:TIME])*24*60*60 .- mspl*60)
	ylims!(-0.1,0.1)
	
	
	fg5
end

# ╔═╡ 0e1b858a-3626-4ca8-a295-4cb734f7aad3
md"""
- There are gaps, shorter interruptions and a drift in in interval between an observation and the next.

- Let's now conpute the LS periodogram for the light-curve.
"""

# ╔═╡ 0376c5a9-45d9-41be-88ba-307f0d88e5ac
begin
	lsper = lombscargle(tblex[!,:TIME],tblex[!,:SAP_FLUX]/1e6,tblex[!,:SAP_FLUX_ERR]/1e6,minimum_frequency=0,maximum_frequency=200)
	
	pmax = findmaxperiod(lsper,[0,1])[1] * 24
	
	
	fg6 = Figure()
	
	axfg6 = Axis(fg6[1,1],
	    ylabel="Lomb-Scargle power",
	    title="Light Curve Power Spectrum",
	    xlabel=L"Frequency (hour$^{-1}$)"
	    )
	
	lines!(lsper.freq / 24, lsper.power)
	xlims!(0, 2.3)
	ylims!(-0.05, 1.0)
	
	text!(0.1, 0.8, text="Best period = "*string(round(pmax,digits=2))*" hours")
	text!(1.5, 0.8, text="Aliased Region")
	
	vspan!(0.5/(mspl/60),2.3, color = (:grey,0.1))
	
	fg6
end

# ╔═╡ 9d1c4f33-5017-461f-855c-97f6625b9612
md"""
- There is a well defined maximu and, due to the "quasi-regular" sampling even an aliased region.

- Finaly, let's check whether te phased curve is reasonable.
"""

# ╔═╡ da662558-149a-4917-be51-da7245be32d0
begin
	phase2 = (tblex[!,:TIME] ./ (pmax/24)) .% 2
	
	fg7 = Figure()
	
	axfg7 = Axis(fg7[1,1],
	    ylabel=L"SAP flux ($10^6 e^-/s$)",
	    title="Phased Light Curve",
	    xlabel="Phase"
	    )
	
	scatter!(phase2,tblex[!,:SAP_FLUX]/1e6)
	errorbars!(phase2,tblex[!,:SAP_FLUX]/1e6,tblex[!,:SAP_FLUX_ERR]/1e6)
	
	fg7
end

# ╔═╡ b8d539e3-35fb-4a0b-a24f-ba09e8083944
md"""
- The obtained light curve confirms the clsassification of this object as a RR-Lyrae variable.
"""

# ╔═╡ 62819b47-de68-4006-8589-0752ebc4c690
md"""
## Reference & Material

Material and papers related to the topics discussed in this lecture.

- [Palaversa et al. (2013) - "Exploring the Variable Sky with LINEAR. III. Classification of Periodic Light Curves”](https://ui.adsabs.harvard.edu/abs/2013AJ....146..101P/abstract)
"""

# ╔═╡ abc29174-e440-4d1d-9bc6-35746b6ffeab
md"""
## Further Material

Papers for examining more closely some of the discussed topics.

- [GCVS - General Catalog of Variable Stars](https://heasarc.gsfc.nasa.gov/W3Browse/all/gcvs.html)
"""

# ╔═╡ a4ad52ac-73d6-4e4e-9593-dfbcbb9ea800
cm"""
### Credits
***

This notebook contains material obtained from [https://github.com/jakevdp/PracticalLombScargle/blob/master/figures/LINEAR_Example.ipynb](https://github.com/jakevdp/PracticalLombScargle/blob/master/figures/LINEAR_Example.ipynb) and [https://github.com/jakevdp/PracticalLombScargle/blob/master/figures/Kepler.ipynb](https://github.com/jakevdp/PracticalLombScargle/blob/master/figures/Kepler.ipynb).
"""

# ╔═╡ b9c07a7b-0c1c-4b13-9cc3-812964e93ef5
cm"""
## Course Flow

<table>
  <tr>
    <td>Previous lecture</td>
    <td>Next lecture</td>
  </tr>
  <tr>
    <td><a href="./open?path=Lectures/Lecture - Lomb-Scargle/Lecture-Lomb-Scargle.jl">Lecture about irregular sampling</a></td>
    <td><a href="./open?path=Lectures/Lecture - Time Domain%20Analysis/Lecture-Time-Domain.jl">Lecture about time domain analysis</a></td>
  </tr>
 </table>


"""

# ╔═╡ edde2429-3894-42d2-b1c7-9ed32c084600
md"""
**Copyright**

This notebook is provided as [Open Educational Resource](https://en.wikipedia.org/wiki/Open_educational_resources). Feel free to use the notebook for your own purposes. The text is licensed under [Creative Commons Attribution 4.0](https://creativecommons.org/licenses/by/4.0/), the code of the examples, unless obtained from other properly quoted sources, under the [MIT license](https://opensource.org/licenses/MIT). Please attribute the work as follows: *Stefano Covino, Time Domain Astrophysics - Lecture notes featuring computational examples, 2025*.
"""

# ╔═╡ Cell order:
# ╟─202d87bd-198a-4394-87d1-a4121daea4cf
# ╟─8353da7d-8182-4a2a-b635-3d9088b3cd24
# ╠═235df63c-0786-42f1-a491-017e67a02698
# ╠═bf2a2828-e952-4e3c-b7bc-aa632262dd30
# ╠═d5a4a4fc-c52b-42c7-ac4e-387ac95f848b
# ╟─fe157ea7-b7c5-4d46-bf1e-a7f69aa8edc3
# ╟─2fe17714-740a-4c8c-8360-b953abf6caf9
# ╟─db391222-d208-42b0-b6f4-eb07e906ad48
# ╟─588828b5-056f-4deb-a2b2-b76035140359
# ╟─9c7b84ca-6b52-43a0-b861-7dd93a0ad93c
# ╟─cec98a24-7e9d-4fe7-a716-f6f81d695028
# ╟─7d835a8b-9075-4d5d-b4cc-ae6cd6b57e27
# ╟─b78f97e9-4b2e-4a8f-a692-9520a2aa81f4
# ╠═11d4292c-42eb-46c1-86f2-88107c25895b
# ╠═61b4ac2f-d559-4895-a628-88cc4b94c683
# ╠═a9125f9d-a9cd-4bfb-95ef-fb8645ae9ff0
# ╟─f73d6e54-031d-4a89-a7a3-3a4c989fc6c6
# ╠═f659d524-8f09-4c82-b87e-bf0cabf19e62
# ╠═ed624515-a93b-45b7-bcab-ae33144281c2
# ╟─f1a6ff9a-d114-4e38-8581-5ed48e78af82
# ╠═d820d95f-d627-4ead-8359-60a3c97ed569
# ╟─1029bc93-7403-4081-b802-5a5664a494c5
# ╠═16df63c3-10c7-4c96-8da3-37e3de4293bf
# ╟─eb548a4e-2af3-4060-adbc-b70565294879
# ╠═58555d35-ff75-4371-b307-7d221f3e7669
# ╟─ef9a0717-9637-4e74-8a49-ed320099fa78
# ╠═b217b22b-b649-4f2c-a347-a632c1cf679d
# ╟─52a3043a-8ca6-49ef-b468-9d6f97c0bdb4
# ╠═bad3dda4-e865-454d-8ffb-a6f34a17addf
# ╟─7da3e53b-75b1-4f88-8a4f-337941a68d2c
# ╠═69e812b8-1ea1-4500-9724-722f2ba6bf65
# ╠═f7dd869f-5ca0-4586-baff-01b07eeb66c1
# ╟─5a3aaddb-e4bc-48b2-bb1e-6c0a371a03a7
# ╠═c7fdf5aa-4981-4ae8-9793-42c848241f34
# ╟─0e1b858a-3626-4ca8-a295-4cb734f7aad3
# ╠═0376c5a9-45d9-41be-88ba-307f0d88e5ac
# ╟─9d1c4f33-5017-461f-855c-97f6625b9612
# ╠═da662558-149a-4917-be51-da7245be32d0
# ╟─b8d539e3-35fb-4a0b-a24f-ba09e8083944
# ╟─62819b47-de68-4006-8589-0752ebc4c690
# ╟─abc29174-e440-4d1d-9bc6-35746b6ffeab
# ╟─a4ad52ac-73d6-4e4e-9593-dfbcbb9ea800
# ╟─b9c07a7b-0c1c-4b13-9cc3-812964e93ef5
# ╟─edde2429-3894-42d2-b1c7-9ed32c084600
